//------Log----
//Created by Rudy Rucker April 16, 1997.

#include "glock3.hpp"
#include "random.h"  //for the randomize functions
#include "gosper3.rh"
#include "error2.hpp"
#include "permute.hpp" //For PermuteBuffer
#include "resource.h"
#include <math.h>  //for cos and sin
#define TRIGMULT 1000.0
// For converting cos and sin values into big integer values (not over 32K).
#define LENGTHMULT 100.0
//For converting lengths into big integer values (not bigger than 32K).
extern HWND main_hwnd;
extern Frame main_frame;
static PermuteBuffer permute_buffer; //Keep this guy around to help Permute.

VectorAndPixel VectorAndPixel::operator=(const VectorAndPixel &a)
{   //Don't bother with copying the __point
	_vector2 = a._vector2;
	return *this;
}

BOOL VectorAndPixel::RealToPixel(const Frame &frame)
{
	BOOL success;
	int ix, iy;
	success = frame.RealToPixel(_vector2.x(), _vector2.y(), &ix, &iy);
	_point.Set(ix,iy);
	return success;
}

VectorAndPixel operator+(const VectorAndPixel &a, const VectorAndPixel &b)
{
	return VectorAndPixel(a._vector2 + b._vector2);
}

VectorAndPixel operator*(Real f, const VectorAndPixel &a)
{
	return VectorAndPixel(f*a._vector2);
}

//=======================GosperHand======================
void GosperHand::SetLength(Real length)
{
	_length = CLAMP(length, -1000.0, 1000.0);
}

void GosperHand::SetPhase(Real phase)
{
	_phase = CLAMP(phase, -TWO_PI, TWO_PI);
}

void GosperHand::SetRate(Real rate)
{
	_rate = CLAMP(rate, -1000.0, 1000.0);
}

void GosperHand::SetColor(COLORREF color)
{
	HDC hdc = GetDC(main_hwnd);
	_color = GetNearestColor(hdc, color);
	ReleaseDC(main_hwnd, hdc);
}

void GosperHand::Randomize(int randomizeflags)
{
	if (randomizeflags & RAND_COLOR)
	{
			int rgb = Random(3);
			switch(rgb)
			{
				case 0:
					_color = RGB(128+Random(128),
						Random(2)?256:Random(256),
						Random(2)?256:Random(256));
					break;
				case 1:
					_color = RGB(Random(2)?256:Random(256),
						128+Random(128),
						Random(2)?256:Random(256));
					break;
				case 2:
					_color = RGB(Random(2)?256:Random(256),
						Random(2)?256:Random(256),
						128+Random(128));
					break;
			}
		HDC hdc = GetDC(main_hwnd);
		_color = GetNearestColor(hdc, _color);
		ReleaseDC(main_hwnd, hdc);
	}
	if (randomizeflags & RAND_LENGTH)
		_length = RAND_MIN_LENGTH +
			Randomreal()*(RAND_MAX_LENGTH - RAND_MIN_LENGTH);
	if (randomizeflags & RAND_PHASE)
		_phase = Randomreal()*TWO_PI;
	if (randomizeflags & RAND_RATE)
		_rate = Randomsign() * ( 1 + Random(RAND_MAX_RATE));
}

//=======================GosperClock======================
GosperClock::GosperClock(int ihand_count):
_t(0.0), _dt(START_TIME_STEP), _max_hand_count(START_MAX_HAND_COUNT),
_hand_count(ihand_count), _handshape(VD_DISK), _writemode(TRUE), _edit_hand_id(0),
_gosper_curve(GC_LINE)
{
	_hand = new GosperHand[_max_hand_count];
	if (!_hand)
		ErrorBoxAndBail("Out of Memory!", "GosperClock::GosperClock");
	for (int i=0; i< _max_hand_count; i++)
	{
	//Use the rates 1,-1,2,-2,3,-3,... as i goes 0,1,2,3,4,5,...
	//(i/2)+1 will give the right absolute values becuase of round-off.
	//Look at i%2 to check even or odd cases.  (Remember % means modulo).
		_hand[i]._rate = (i%2==0)?((i/2)+1):-((i/2)+1);
		_hand[i].Randomize(RAND_LENGTH|RAND_PHASE|RAND_COLOR);
	}
	_tip = new VectorAndPixel[_max_hand_count];
	if (!_tip)
		ErrorBoxAndBail("Out of Memory!", "GosperClock::GosperClock");
	_mark_count = (int)((TWO_PI)/_dt);
	_mark = new VectorAndPixel[_mark_count];
	if (!_mark)
		ErrorBoxAndBail("Out of Memory!", "GosperClock::GosperClock");
	Reset(); //Initializes the _tip[i], _mark[0], and _mark_index.
}

GosperClock::~GosperClock()
{
	delete [] _hand;
	delete [] _tip;
	delete [] _mark;
}

void GosperClock::SetRadius(Frame &frame)
{
	Real radius = 0.2; //To leave some room at the edge.

	for (int i=0; i< _hand_count; i++)
		radius += fabs(_hand[i]._length);
	frame.SetRealWindow(_center.x() - radius, _center.y() - radius,
		_center.x() + radius, _center.y() + radius);
}

void GosperClock::Reset()
{
	_t = 0.0;
	_tip[0] = _center + _hand[0]._length *
		VectorAndPixel(cos(_hand[0]._phase), sin(_hand[0]._phase));
	for (int i=1; i< _hand_count; i++)
	{
		_tip[i] = _tip[i-1] + _hand[i]._length *
			VectorAndPixel(cos(_hand[i]._phase), sin(_hand[i]._phase));
	}
	//Now find the smallest non-zero hand-rate magnitude
	Real min_rate = 1000000.0;
	for (i=0; i< _hand_count; i++)
		if (_hand[i]._rate && fabs(_hand[i]._rate) < min_rate)
			min_rate = fabs(_hand[i]._rate);
	//Use min_rate to compute the time for one full period.
	if (min_rate < SMALL_REAL)
		min_rate = SMALL_REAL;
	if (min_rate > 1.0)
		min_rate = 1.0;
	Real period_time = TWO_PI / min_rate;
	_mark_count = (int)(period_time/_dt);
	if (!_mark_count)
		_mark_count = 1;
	//
	if (_mark)
		delete [] _mark;
	_mark = new VectorAndPixel[_mark_count];
	if (!_mark)
		ErrorBoxAndBail("Out of _mark Memory!", "GosperClock::Reset");
	_mark[0] = _tip[_hand_count - 1];
	_mark_index = 1;
}

void GosperClock::SetHandshape(int handshape)
{
	_handshape = handshape;
}

void GosperClock::Randomize(int randomizeflags)
{
	if (randomizeflags & RAND_COUNT)
  {
		_hand_count = RAND_MIN_COUNT + Random(RAND_MAX_COUNT -
			RAND_MIN_COUNT);
     SetHandCount(_hand_count);
  }
	for (int i = 0; i<_hand_count; i++)
		_hand[i].Randomize(randomizeflags);
	if (randomizeflags)
		Reset();
}

void GosperClock::SetGosperCurve(int code)
{
	int k;
	Real sides;

	switch(code)
	{
		case GC_LINE:

//LINE?
			for (k=0; k< _max_hand_count; k++)
			{
				_hand[k]._rate = (k%2==0)?((k/2)+1):-((k/2)+1);
				_hand[k]._length = 1.0;
				_hand[k]._phase= 0.0;
			}
			_gosper_curve = code;
			break;

		case GC_SMOOTH_TRIANGLE:

//TRIANGLE
			for (k=0; k< _max_hand_count; k++)
			{
				_hand[k]._length = pow((1 + sin(PI*(4*k+1)/6))/(k+1), 2);
				_hand[k]._rate = (k+1)*(k%3==1?-1:1);
				_hand[k]._phase= 0.0;
			}
			_gosper_curve = code;
			break;

		case GC_TRIANGLE:
 //GIBBS TRIANGLE
			for (k=0; k< _max_hand_count; k++)
			{
				_hand[k]._length = (2*cos((PI*k)/3.0) + (k%2?-1:1))/(1 + (k/2));
				_hand[k]._rate = (k%2?-1:1)*(1 + (k/2));
				_hand[k]._phase= 0.0;
			}
			_gosper_curve = code;
			break;
		
		case GC_SQUARE:
			//SQUARE
			for (k=0; k< _max_hand_count; k++)
			{
				_hand[k]._length = ((k%2)?-1.0:1.0)/(2.0*k+1);
				_hand[k]._rate = (int)(((k%2)?-1.0:1.0)*(2.0*k+1));
				_hand[k]._phase= 0;
			}
			_gosper_curve = code;
			break;

		case GC_PENTAGON:
			//NGON
			sides = 5.0;
			for (k=0; k< _max_hand_count; k++)
			{
				int ceil = (k+1)/2;
				_hand[k]._length = (ceil%2?-1:1)/((k%2?-1:1)*ceil + (1.0/sides));
				_hand[k]._rate = (k%2?-1:1) * ceil + (1.0/sides);
				_hand[k]._phase= 0.0;
			}
			_gosper_curve = code;
			break;
		case GC_SEVEN:
			//NGON
			sides = 7.0;
			for (k=0; k< _max_hand_count; k++)
			{
				int ceil = (k+1)/2;
				_hand[k]._length = (ceil%2?-1:1)/((k%2?-1:1)*ceil + (1.0/sides));
				_hand[k]._rate = (k%2?-1:1) * ceil + (1.0/sides);
				_hand[k]._phase= 0.0;
			}
			_gosper_curve = code;
			break;
		case GC_TWELVE:
			//NGON
			sides = 12.0;
			for (k=0; k< _max_hand_count; k++)
			{
				int ceil = (k+1)/2;
				_hand[k]._length = (ceil%2?-1:1)/((k%2?-1:1)*ceil + (1.0/sides));
				_hand[k]._rate = (k%2?-1:1) * ceil + (1.0/sides);
				_hand[k]._phase= 0.0;
			}
			_gosper_curve = code;
			break;
	}
	Reset();
}


void GosperClock::Permute()
{
	permute_buffer.permute(_hand_count);
	GosperHand *new_hand;
	new_hand = new GosperHand[_max_hand_count];
	if (!new_hand)
		ErrorBoxAndBail("Out of Memory!", "GosperClock::Permute");
	for (int i=0; i< _hand_count; i++)
		new_hand[i] = _hand[permute_buffer[i]];
	for (i=_hand_count; i< _max_hand_count; i++)
		new_hand[i] = _hand[i];
	delete [] _hand;
	_hand = new_hand;
	Reset();
}

void GosperClock::Tick()
{
	Real handangle;

	_t += _dt;
	for (int i=0; i< _hand_count; i++)
	{
	  	handangle = _hand[i]._phase + _hand[i]._rate*_t;
		if (!i)
			_tip[i] = _center + _hand[0]._length *
				VectorAndPixel(cos(handangle), sin(handangle));
		else
			_tip[i] = _tip[i-1] + _hand[i]._length *
				VectorAndPixel( cos(handangle), sin(handangle));
	}
	if (_mark_index < _mark_count)
	{
		_mark[_mark_index] = _tip[_hand_count-1];
		_mark_index++;
	}
}

void GosperClock::RealToPixel(const Frame &frame)
{
	_center.RealToPixel(frame);
	for (int i=0; i< _hand_count; i++)
		_tip[i].RealToPixel(frame);
	for (i=0; i< _mark_index; i++)  //Don't always need to do all
		_mark[i].RealToPixel(frame);
}

void GosperClock::Draw(HDC hdc)
{
	HPEN hpen;
	HBRUSH hbrush;
	int i;
	int left, top, right, bottom, startx, starty;
	VectorAndPixel centeri;

//Draw the curve
	hpen = CreatePen(PS_SOLID, 2, RGB(255,255,255));
	SelectObject(hdc,hpen);
	MoveToEx(hdc,_mark[0].pixx(),_mark[0].pixy(), NULL);
	for (i=0; i< _mark_index; i++)
		LineTo(hdc,_mark[i].pixx(),_mark[i].pixy());
	if (i == _mark_count)
		LineTo(hdc,_mark[0].pixx(),_mark[0].pixy());
	SelectObject(hdc, GetStockObject(WHITE_PEN));
	DeleteObject(hpen);
//Draw the hands
	switch (_handshape)
	{
		case VD_HAND:
			MoveToEx(hdc, _center.pixx(), _center.pixy(), NULL);
			for (i=0; i< _hand_count; i++)
			{
				hpen = CreatePen(PS_SOLID, 3, _hand[i]._color);
				SelectObject(hdc, hpen);
				LineTo(hdc, _tip[i].pixx(), _tip[i].pixy());
				SelectObject(hdc, GetStockObject(WHITE_PEN));
				DeleteObject(hpen);
			}
			break;
		case VD_PIE:
			SetROP2(hdc, R2_XORPEN);
			for (i=0; i< _hand_count; i++)
			{
				hbrush = CreateSolidBrush(_hand[i]._color);
				SelectObject(hdc, hbrush);
				SelectObject(hdc, GetStockObject(NULL_PEN));
				if (!i)
					centeri = _center;
				else
					centeri = _tip[i-1];
				main_frame.RealToPixel(centeri.x()-_hand[i]._length,
					centeri.y()+_hand[i]._length, &left, &top);
				main_frame.RealToPixel(centeri.x()+_hand[i]._length,
					centeri.y()-_hand[i]._length, &right, &bottom);
				main_frame.RealToPixel(
					centeri.x() + _hand[i]._length * cos(_hand[i]._phase),
					centeri.y() + _hand[i]._length * sin(_hand[i]._phase),
					&startx, &starty);
				if (_hand[i]._rate > 0) //counterclockwise use main pie
					Pie(hdc, left, top, right, bottom, startx, starty, _tip[i].pixx(),
						_tip[i].pixy());
				else //clockwise use complemtary part of the pie.
					Pie(hdc, left, top, right, bottom, _tip[i].pixx(),
						_tip[i].pixy(), startx, starty);
				SelectObject(hdc, GetStockObject(BLACK_BRUSH));
				DeleteObject(hbrush);
			}
			SetROP2(hdc, R2_COPYPEN);
			break;
		case VD_DISK:
			SetROP2(hdc, R2_XORPEN);
			for (i=0; i< _hand_count; i++)
			{
				hbrush = CreateSolidBrush(_hand[i]._color);
				SelectObject(hdc, hbrush);
				SelectObject(hdc, GetStockObject(NULL_PEN));
				if (!i)
					centeri = _center;
				else
					centeri = _tip[i-1];
				main_frame.RealToPixel(centeri.x()-_hand[i]._length,
					centeri.y()+_hand[i]._length, &left, &top);
				main_frame.RealToPixel(centeri.x()+_hand[i]._length,
					centeri.y()-_hand[i]._length, &right, &bottom);
				main_frame.RealToPixel(
					centeri.x() + _hand[i]._length * cos(_hand[i]._phase),
					centeri.y() + _hand[i]._length * sin(_hand[i]._phase),
					&startx, &starty);

				Ellipse(hdc, left, top, right, bottom);
				SelectObject(hdc, GetStockObject(BLACK_BRUSH));
				DeleteObject(hbrush);
			}
			SetROP2(hdc, R2_COPYPEN);
			SelectObject(hdc, GetStockObject(WHITE_PEN));
			MoveToEx(hdc, _center.pixx(), _center.pixy(), NULL);
			for (i=0; i< _hand_count; i++)
				LineTo(hdc, _tip[i].pixx(), _tip[i].pixy());
			break;
	}
}

BOOL GosperClock::SetHandCount(int n)
{
	if (n <= 0)
		return FALSE;
	if (n<=_max_hand_count)
	{
		_hand_count = n;
	}
	else
	{
		int old_max_hand_count = _max_hand_count;
		_max_hand_count = _hand_count = n;
		GosperHand *new_hand = new GosperHand[_max_hand_count];
		VectorAndPixel *new_tip = new VectorAndPixel[_max_hand_count];
		if (!new_hand || !new_tip)
		{   //Not enough room to make more hands.
			_max_hand_count = old_max_hand_count;
			return FALSE;
		}
		for (int k=0; k<old_max_hand_count; k++)
			new_hand[k] = _hand[k];
		for (k=old_max_hand_count; k<_max_hand_count; k++)
		{
			new_hand[k]._rate = (k%2==0)?((k/2)+1):-((k/2)+1);
			new_hand[k].Randomize(RAND_HAND);
		}
		delete [] _hand;
		_hand = new_hand;
		delete [] _tip;
		_tip = new_tip;
	}
	if (_edit_hand_id >= _hand_count)
		_edit_hand_id = _hand_count-1;
	Reset();
	return TRUE;
}

BOOL GosperClock::ChangeHandCount(int updown)
{
	return SetHandCount(_hand_count + updown);
}

Real GosperClock::EditHandLength(){return _hand[_edit_hand_id]._length;}
Real GosperClock::EditHandRate(){return _hand[_edit_hand_id]._rate;}
Real GosperClock::EditHandPhase(){return _hand[_edit_hand_id]._rate;}
COLORREF GosperClock::EditHandColor(){return _hand[_edit_hand_id]._color;}

#define MIN_DT 0.003
#define MAX_DT 6.0
void GosperClock::SetDt(Real newdt)
{
	_dt = newdt;
	if (_dt < MIN_DT)
		_dt = MIN_DT;
	if (_dt > MAX_DT)
		_dt = MAX_DT;
	Reset();
}

void GosperClock::SetEditHandID(int n)
{
	if (n >= _hand_count || n < 0)
		return;
	_edit_hand_id = n;
}

void GosperClock::SetHandLength(int n, Real length)
{
	if (n > _hand_count || n < 0)
		return;
	_hand[n].SetLength(length);
}

void GosperClock::SetHandPhase(int n, Real phase)
{
	if (n > _hand_count || n < 0)
		return;
	_hand[n].SetPhase(phase);
}

void GosperClock::SetHandRate(int n, Real rate)
{
	if (n > _hand_count || n < 0)
		return;
	_hand[n].SetRate(rate);
}

void GosperClock::SetHandColor(int n, COLORREF color)
{
	if (n > _hand_count || n < 0)
		return;
	_hand[n].SetColor(color);
}


BOOL GosperClock::Save(char *filename)
{
	ofstream ofs;
	//We don't need to check if this is an overwrite as
	//GetOpenFileName has done this for us.
	ofs.open(filename, ios::out); //ready to writer or overwrite 
	ofs << *this;
	ofs.close();
	return TRUE;
}

BOOL GosperClock::Open(char* filename)
{
	ifstream ifs;

	ifs.open(filename, ios::in|ios::nocreate);
	if (ifs.fail())
	{
		MessageBox( main_hwnd,
			(LPSTR)"Can't open file!",
			(LPSTR)filename,
			MB_OK | MB_ICONEXCLAMATION );
		ifs.close();
		return FALSE;
	}
	ifs >> *this;
	ifs.close();
	return TRUE;
}

ifstream& operator>>(ifstream& ifs, GosperClock &gc)
{
	char dummybuf[80];
	Real readreal;
	int readint;
	int file_hand_count;
	//COLORREF readcolor;
		/*We don't copy the values right into the gc
		because they need to be set using the mutator.*/

	// Read the line "\n Timestep is %"
	ifs.get();  // read off the "\n" (end of line)
	 //Read two words and a value
	ifs >> dummybuf >> dummybuf  >> readreal;
	gc.SetDt(readreal);
	// Read the line "\n File Handcount is %"
	ifs.get();  // read off the "\n" (end of line)
	//Read three words and a value
	ifs >> dummybuf >> dummybuf >> dummybuf >> readint;
	file_hand_count = readint;
	gc.SetHandCount(readint); //To make sure that _max_hand_count >= file_hand_count.
	// Read the line "\n Active Handcount is %"
	ifs.get();  // read off the "\n" (end of line)
	//Read three words and a value
	ifs >> dummybuf >> dummybuf >> dummybuf >> readint;
	gc.SetHandCount(readint);
	for (int i=0; i<file_hand_count; i++)
	{
		// Read the line "\n Hand %"
		ifs.get();  // read off the "\n" (end of line)
		//Read a word and a value
		ifs >> dummybuf >> readint;
		// Read the line "\n Length %"
		ifs.get();  // read off the "\n" (end of line)
		//Read a word and a value
		ifs >> dummybuf >> readreal;
		gc.SetHandLength(i, readreal);
		// Read the line "\n Rate %"
		ifs.get();  // read off the "\n" (end of line)
		//Read a word and a value
		ifs >> dummybuf >> readreal;
		gc.SetHandRate(i, readreal);
			// Read the line "\n Phase %"
		ifs.get();  // read off the "\n" (end of line)
		//Read a word and a value
		ifs >> dummybuf >> readreal;
		gc.SetHandPhase(i, readreal);
	}		
	gc.Reset();
	return ifs;
}

ofstream& operator<<(ofstream& ofs, GosperClock &gc)
{ /* Since we look at private _params and _textcolor we have to be friend of
	both GDIToolbox and of GDIParams. We could instead use the accessor and
	not have to be a friend.*/
	ofs << endl << "Timestep is " << gc.dt();
	ofs << endl << "File handcount is " << gc._max_hand_count;
	ofs << endl << "Active handcount is " << gc._hand_count;
	for (int i=0; i<gc._max_hand_count; i++)
	{
		ofs << endl << "Hand " << i;
		ofs << endl << "Length " << gc._hand[i]._length;
		ofs << endl << "Rate " << gc._hand[i]._rate;
		ofs << endl << "Phase " << gc._hand[i]._phase;
	}
	return ofs;
}
